'use client'

import { useState, useEffect } from 'react'
import { useParams, useRouter } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import Link from 'next/link'

export default function EditarClientePage() {
  const params = useParams()
  const router = useRouter()
  const [loading, setLoading] = useState(true)
  const [guardando, setGuardando] = useState(false)
  const [formData, setFormData] = useState({
    nombre: '',
    email: '',
    telefono: '',
    empresa: '',
    direccion: '',
    rfc: '',
    notas: '',
    activo: true,
  })

  useEffect(() => {
    if (params.id) {
      fetchCliente(params.id as string)
    }
  }, [params.id])

  async function fetchCliente(id: string) {
    try {
      const { data, error } = await supabase
        .from('clientes')
        .select('*')
        .eq('id', id)
        .single()

      if (error) throw error
      
      setFormData({
        nombre: data.nombre || '',
        email: data.email || '',
        telefono: data.telefono || '',
        empresa: data.empresa || '',
        direccion: data.direccion || '',
        rfc: data.rfc || '',
        notas: data.notas || '',
        activo: data.activo,
      })
    } catch (error) {
      console.error('Error fetching cliente:', error)
    } finally {
      setLoading(false)
    }
  }

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault()
    
    // Validar que al menos nombre o empresa esté lleno
    if (!formData.nombre && !formData.empresa) {
      alert('Debes ingresar al menos un nombre o una empresa')
      return
    }
    
    setGuardando(true)

    try {
      const { error } = await supabase
        .from('clientes')
        .update({
          nombre: formData.nombre || null,
          email: formData.email || null,
          telefono: formData.telefono || null,
          empresa: formData.empresa || null,
          direccion: formData.direccion || null,
          rfc: formData.rfc || null,
          notas: formData.notas || null,
          activo: formData.activo,
        })
        .eq('id', params.id)

      if (error) throw error

      router.push(`/clientes/${params.id}`)
    } catch (error: any) {
      console.error('Error updating cliente:', error)
      alert(`Error al actualizar el cliente: ${error.message}`)
    } finally {
      setGuardando(false)
    }
  }

  if (loading) {
    return <div className="text-center p-8">Cargando...</div>
  }

  return (
    <div className="max-w-2xl mx-auto">
      <div className="mb-4">
        <Link href={`/clientes/${params.id}`} className="text-blue-600 hover:underline">← Volver</Link>
      </div>
      <h1 className="text-3xl font-bold mb-6">Editar Cliente</h1>
      <form onSubmit={handleSubmit} className="bg-white p-6 rounded-lg shadow space-y-4">
        <div>
          <label className="block text-sm font-medium mb-1">Nombre</label>
          <input
            type="text"
            value={formData.nombre}
            onChange={(e) => setFormData({ ...formData, nombre: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Nombre completo (opcional si solo es empresa)"
          />
        </div>
        <div>
          <label className="block text-sm font-medium mb-1">Email</label>
          <input
            type="email"
            value={formData.email}
            onChange={(e) => setFormData({ ...formData, email: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="correo@ejemplo.com"
          />
        </div>
        <div>
          <label className="block text-sm font-medium mb-1">Teléfono</label>
          <input
            type="tel"
            value={formData.telefono}
            onChange={(e) => setFormData({ ...formData, telefono: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="+56 9 1234 5678"
          />
        </div>
        <div>
          <label className="block text-sm font-medium mb-1">Empresa</label>
          <input
            type="text"
            value={formData.empresa}
            onChange={(e) => setFormData({ ...formData, empresa: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="Nombre de la empresa (opcional si hay nombre personal)"
          />
        </div>
        <div>
          <label className="block text-sm font-medium mb-1">Dirección</label>
          <textarea
            value={formData.direccion}
            onChange={(e) => setFormData({ ...formData, direccion: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            rows={3}
            placeholder="Dirección completa"
          />
        </div>
        <div>
          <label className="block text-sm font-medium mb-1">RUT</label>
          <input
            type="text"
            value={formData.rfc}
            onChange={(e) => setFormData({ ...formData, rfc: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            placeholder="RUT123456789"
          />
        </div>
        <div>
          <label className="block text-sm font-medium mb-1">Notas</label>
          <textarea
            value={formData.notas}
            onChange={(e) => setFormData({ ...formData, notas: e.target.value })}
            className="w-full border rounded px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
            rows={3}
            placeholder="Notas adicionales sobre el cliente"
          />
        </div>
        <div className="flex items-center">
          <input
            type="checkbox"
            checked={formData.activo}
            onChange={(e) => setFormData({ ...formData, activo: e.target.checked })}
            className="mr-2"
          />
          <label className="text-sm font-medium">Cliente activo</label>
        </div>
        <div className="flex gap-4 pt-4">
          <button
            type="submit"
            disabled={guardando}
            className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700 disabled:opacity-50 transition"
          >
            {guardando ? 'Guardando...' : 'Guardar Cambios'}
          </button>
          <button
            type="button"
            onClick={() => router.back()}
            className="bg-gray-300 text-gray-700 px-6 py-2 rounded hover:bg-gray-400 transition"
          >
            Cancelar
          </button>
        </div>
      </form>
    </div>
  )
}
